//------------------------------------------------------------------------------
// File: client_statGui.cs
// This file contains the methods that will handle the stat screen hud elements
// Author: Matthew Rudge
//------------------------------------------------------------------------------

//-Stat Gui Globals-------------------------------------------------------------
$HappyModString[$HMOD_POPULATION]   = "%s Population:";
$HappyModString[$HMOD_HUNGER]       = "Hungry Civilians:";
$HappyModString[$HMOD_UNEMPLOYED]   = "Unemployed Civilians:";
$HappyModString[$HMOD_HOMELESS]     = "Homeless Civilians:";
$HappyModString[$HMOD_DRUNK]        = "Drunk Civilians:";
$HappyModString[$HMOD_TOUGHSLINGER] = "Hired Tough Gunslingers:";
$HappyModString[$HMOD_NICESLINGER]  = "Hired Nice Gunslingers:";
$HappyModString[$HMOD_TOUGHSHERIFF] = "Hired Tough Sheriff:";
$HappyModString[$HMOD_NICESHERIFF]  = "Hired Nice Sheriff:";
$HappyModString[$HMOD_BUILDINGBLDG] = "%s:";
$HappyModString[$HMOD_DAMAGEBLDG]   = "Damaged Buildings:";
$HappyModString[$HMOD_PRODUCE]      = "Food Varieties:";
$HappyModString[$HMOD_BANDITATTACK] = "Bandit Attacks:";
$HappyModString[$HMOD_ANIMALATTACK] = "Animal Attacks:";
$HappyModString[$HMOD_TAXES]     = "%s Taxes:";
$HappyModString[$HMOD_PARADE]       = "Parade:";
$HappyModIconHappy = "assets/client/ui/icons/happyicon.png";
$HappyModIconMad = "assets/client/ui/icons/madicon.png";
$HappyModCount = 0;

$ObjTypeStatCount = 0;

// Resource stats
$StatGuiRsrcTextCount   = 4;
$StatGuiRsrc[0]         = $Resource::Gold;
$StatGuiRsrc[1]         = $Resource::Wood;
$StatGuiRsrc[2]         = $Resource::Food;
$StatGuiRsrc[3]         = $Resource::Water;
$StatGuiRsrcProdText[0] = GoldProdStat;
$StatGuiRsrcProdText[1] = WoodProdStat;
$StatGuiRsrcProdText[2] = FoodProdStat;
$StatGuiRsrcProdText[3] = WaterProdStat;
$StatGuiRsrcConText[0]  = GoldConStat;
$StatGuiRsrcConText[1]  = WoodConStat;
$StatGuiRsrcConText[2]  = FoodConStat;
$StatGuiRsrcConText[3]  = WaterConStat;
//------------------------------------------------------------------------------

//-Dialog Methods---------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
//! Opens the stat gui
////////////////////////////////////////////////////////////////////////////////
function openStatGui()
{
   // if the game menu is already loaded, exit
   if ($GameMenuLoaded == true)
   {
      return;
   }
   
   slgStackPause();
   
   // present the stat page
   slgRenderWorld(false);
   $GameMenuLoaded = true;
   Canvas.pushDialog(StatGui);
   
   populateStatGuiResources();
   populateStatGuiObjects();
   populateStatGuiHappiness();
}

////////////////////////////////////////////////////////////////////////////////
//! Closes the stat gui
////////////////////////////////////////////////////////////////////////////////
function closeStatGui()
{
   Canvas.popDialog(StatGui);
   clearStatGui();
   
   slgRenderWorld(true);
   slgStackUnPause();
   $GameMenuLoaded = false;
}

////////////////////////////////////////////////////////////////////////////////
//! Clears the stat gui of all previous values
////////////////////////////////////////////////////////////////////////////////
function clearStatGui()
{
   // Reset happy modifier gui values
   resetHappyMods();
   
   // Clear object type stats
   resetObjTypeStats();
}

/*
////////////////////////////////////////////////////////////////////////////////
//! Called as a final act before opening stat gui. (Clears happy modifier rows)
////////////////////////////////////////////////////////////////////////////////
function finalizeStatGui()
{
   // For each happy mod do
   for(%i = 0; %i < $HMOD_COUNT; %i++) {
      // If happy mod value is zero then delete mod
      if(getHappyModValue(%i) == 0) {
         deleteHappyMod(%i);
      }
   }
}
*/

////////////////////////////////////////////////////////////////////////////////
//! Called to initialize and open the stat gui (Obtains happy modifiers on 
//! server)
////////////////////////////////////////////////////////////////////////////////
function initializeStatGui()
{
   // clear any existing command states
   csClearTargetState();
   
   // Send out command to get modifier values
   commandToServer('GetHappyMods');
}

////////////////////////////////////////////////////////////////////////////////
//! Populates the stat gui resource section
////////////////////////////////////////////////////////////////////////////////
function populateStatGuiResources()
{
   // Resource counts
   GoldStat.stateUp   = rsGetResourceCount($Resource::Gold);
   WoodStat.stateUp   = rsGetResourceCount($Resource::Wood);
   FoodStat.stateUp   = rsGetResourceCount($Resource::Food);
   WaterStat.stateUp  = rsGetResourceCount($Resource::Water);
   PeopleStat.stateUp = rsGetResourceCount($Resource::People);
   ExpStat.stateUp    = rsGetResourceCount($Resource::Experience);
   
   // Resource caps
   FoodCapStat.stateUp   = rsGetResourceCap($Resource::Food);
   WaterCapStat.stateUp  = rsGetResourceCap($Resource::Water);
   PeopleCapStat.stateUp = rsGetResourceCap($Resource::People);
   
    // Init resource production and consumption
   for(%i = 0; %i < $StatGuiRsrcTextCount; %i++) {
      // Reset to zero
      $StatGuiRsrcProdText[%i].stateUp = 0;
      $StatGuiRsrcConText[%i].stateUp  = 0;
   }
   
   // initialize the gold production stat based on the general
   // store production rates
   if (GeneralStoreProfit.isAvailable() == false)
   {
      return;
   }
   
   // check if general stores are producing any resources
   %gold = 0;
   if (GeneralStorePumpkin.isComplete() == true) %gold += 5;
   if (GeneralStoreSheep.isComplete() == true) %gold += 5;
   if (GeneralStoreHoneyCreate.isComplete() == true) %gold += 10;
   if (%gold == 0)
   {
      return;
   }
   
   // get the general stores
   %list = tsCreateObjList();
   tsFindObjTypeOnMap("General", "bldg", %list);
   
   // count the total resources collected
   %count = %list.getSize();
   for (%index = 0; %index < %count; %index++)
   {
      // if this general store is not on the player's team, skip it
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // if the building is not producing, skip it
      %building = slgqueryinterface(%object, $CID_BUILDING);
      if (%building.getState() != $BuildingState::Production)
      {
         continue;
      }
      
      // if the building has an employee, give it resources
      if (%building.getEmployeeCount() > 0)
      {
         // update the gold production stat
         GoldProdStat.stateUp += %gold;
      }
   }
   tsDestroyObjList(%list);
}

////////////////////////////////////////////////////////////////////////////////
//! Populates the stat gui with information about happiness
////////////////////////////////////////////////////////////////////////////////
function populateStatGuiHappiness()
{
   // Get happiness percentage
   HappyPercentStat.stateUp = rsGetResourceCount($Resource::Happiness);
   
   // initially hide any templates
   HappyStringStatFirst.visible = false;
   HappyValueStatFirst.visible = false;
   HappyIconStatFirst.visible = false;
   HappyStringStatNext.visible = false;
   HappyValueStatNext.visible = false;
   HappyIconStatNext.visible = false;
   
   // initialize the happy bonus
   HappyBonusStat.stateUp = "0";
   
   // load the happy mods
   for (%index = 0; %index < $HMOD_COUNT; %index++)
   {
      commandToServer('FindHappyModData', %index);
   }
   commandToServer('FindHappyModDone');
   
   /*
   // Get overpopulation modifier
   %peopleCnt = rsGetResourceCount($Resource::People);
   %happyMod  = GameResourceData.getTotalPopulationHappiness(%peopleCnt);
   if(%happyMod != 0) {
      setHappyModValue($HMOD_POPULATION, %happyMod);
      setHappyModGui($HMOD_POPULATION);
   }
   
   // Update server sent happy modifiers
   updateHappyModGui($HMOD_BANDITATTACK);
   updateHappyModGui($HMOD_PARADE);
   updateHappyModGui($HMOD_ANIMALATTACK);
   updateHappyModGui($HMOD_TAXES);
   updateHappyModGui($HMOD_BUILDINGBLDG);
   updateHappyModGui($HMOD_PRODUCE);
   updateHappyModGui($HMOD_DAMAGEBLDG);
   */
}

////////////////////////////////////////////////////////////////////////////////
//! Populates the stat gui with information about all the objects
////////////////////////////////////////////////////////////////////////////////
function populateStatGuiObjects()
{
   // Reset combat stats
   NiceSheriffStat.stateUp   = 0;
   ToughSheriffStat.stateUp  = 0;
   DeputyStat.stateUp        = 0;
   NiceSlingerStat.stateUp   = 0;
   ToughSlingerStat.stateUp  = 0;
   FemaleSlingerStat.stateUp = 0;
   MaleSlingerStat.stateUp   = 0;
   
   // Reset character status stats
   EmployedCountStat.stateUp = 0;
   JoblessCountStat.stateUp  = 0;
   HomelessCountStat.stateUp = 0;
   
   // reset the building count stats
   TotalLuxuryBuildingStat.stateUp = 0;
   TotalTownBuildingStat.stateUp = 0;
   
   // Get character list
   %chars = slgGetClientCharacterList();
   if(isObject(%chars)) {
      // For each character do
      %count = %chars.getSize();
      for(%i = 0; %i < %count; %i++) {
         // Get character
         %char = %chars.getID(%i);
         if(!isObject(%char)) {
            continue;
         }
         
         // Character needs to be on player team
         if(!%char.isOnTeam($OST_PLAYER)) {
            continue;
         }
         
         // Populate happiness for character
         populateStatGuiCharacterHappiness(%char);
         populateStatGuiObjectResources(%char);
         populateStatGuiCharacterCount(%char);
      }
   }
   
   // Get building list
   %bldgs = slgGetClientBuildingList();
   if(isObject(%bldgs)) {
      // For each building do
      %count = %bldgs.getSize();
      for(%i = 0; %i < %count; %i++) {
         // Get building
         %bldg = %bldgs.getID(%i);
         if(!isObject(%bldg)) {
            continue;
         }
         
         // Building needs to be on player team
         if(!%bldg.isOnTeam($OST_PLAYER)) {
            continue;
         }
         
         // Building needs to be in a production state
         %cmpBldg = slgQueryInterface(%bldg, $CID_BUILDING);
         if(%cmpBldg.getState() != $BuildingState::Production) {
            continue;
         }
         
         // Populate building list for building
         modifyObjTypeStat(%bldg.getType(), 1);
         populateStatGuiObjectResources(%bldg);
         populateStatGuiBuildingCount(%bldg);
         saveObjTypeStat(%bldg.getType());
      }
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Populates the stat gui happiness for a character
//! \param %char Character to use
////////////////////////////////////////////////////////////////////////////////
function populateStatGuiCharacterHappiness(%char)
{
   %mod = -1;
   %cmpChar = slgQueryInterface(%char, $CID_CHARACTER);
   if(!isObject(%cmpChar)) {
      return;
   }
   
   // Get modifiers for tough and nice characters
   if(%char.isOfType("Nicegunslinger")) {
      %mod = $HMOD_NICESLINGER;
   }
   else if(%char.isOfType("Toughgunslinger")) {
      %mod = $HMOD_TOUGHSLINGER;
   }
   else if(%char.isOfType("Nicesheriff")) {
      %mod = $HMOD_NICESHERIFF;
   }
   else if(%char.isOfType("Toughsheriff")) {
      %mod = $HMOD_TOUGHSHERIFF;
   }
   
   /*
   // Update modifier value for tough and nice characters
   if(%mod != -1) {
      modifyHappyModValue(%mod, %cmpChar.getDatablock().initHappiness);
      setHappyModGui(%mod);
   }
   
   // Check for hunger modifiers
   %hunger = %cmpChar.getHunger();
   if(%hunger != 0) {
      modifyHappyModValue($HMOD_HUNGER, -%hunger);
      setHappyModGui($HMOD_HUNGER);
   }
   */
}

////////////////////////////////////////////////////////////////////////////////
//! Sets the current tally of production and consumption 
////////////////////////////////////////////////////////////////////////////////
function populateStatGuiObjectResources(%object)
{
   if(slgIsCharacter(%object)) {
      %component = slgQueryInterface(%object, $CID_CHARACTER);
      %datablock = %component.getDataBlock();
      FoodConStat.stateUp += %datablock.consumeFood;
   }
   else if(slgIsBuilding(%object)) {
      %cmpBldg = slgQueryInterface(%object, $CID_BUILDING);
      for(%i = 0; %i < $StatGuiRsrcTextCount; %i++) {
         if (%cmpBldg.doesProduce($StatGuiRsrc[%i]) == true)
         {
            %prod = %cmpBldg.getTrueProduction($StatGuiRsrc[%i]);
            $StatGuiRsrcProdText[%i].stateUp += %prod;
         }
         
         %con  = %cmpBldg.getTrueConsumption($StatGuiRsrc[%i]);
         $StatGuiRsrcConText[%i].stateUp  += %con;
      }
      
      if(%object.isOfType("TownHall")) {
         %amount = %cmpBldg.getCurrentTaxAmount(slgGetClientCharacterList());
         GoldProdStat.stateUp += %amount;
      }
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Sets the current tally of various building statistics
////////////////////////////////////////////////////////////////////////////////
function populateStatGuiBuildingCount(%bldg)
{
   // check if the building is a honey pot (if so, ignore it)
   %type = %bldg.getType();
   if (%type $= "Honeypot")
   {
      return;
   }
   
   // check if the building type is a luxury building
   if (%bldg.isLuxuryBuilding() == true)
   {
      TotalLuxuryBuildingStat.stateUp += 1;
   }
   // check if the building type is a town building
   else if (%bldg.isTownBuilding() == true)
   {
      TotalTownBuildingStat.stateUp += 1;
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Sets the current tally of various character statistics
////////////////////////////////////////////////////////////////////////////////
function populateStatGuiCharacterCount(%char)
{
   %cmpChar = slgQueryInterface(%char, $CID_CHARACTER);
   if(isObject(%cmpChar)) {
      // Character is employed
      if(%cmpChar.hasWork()) {
         EmployedCountStat.stateUp += 1;
      }
      // If character can be employed and does not have work
      else if(%cmpChar.isEmployable()) {
         JoblessCountStat.stateUp += 1;
      }
      
      // If character is homeless and needs to find a home
      if(!%cmpChar.hasHome() && %cmpChar.isShelterable()) {
         HomelessCountStat.stateUp += 1;
      }
   }
      
   // Slinger types
   if(%char.isOfType("Malegunslinger")) {
      MaleSlingerStat.stateUp += 1;
   }
   else if(%char.isOfType("Femalegunslinger")) {
      FemaleSlingerStat.stateUp += 1;
   }
   else if(%char.isOfType("Nicegunslinger")) {
      NiceSlingerStat.stateUp += 1;
   }
   else if(%char.isOfType("Toughgunslinger")) {
      ToughSlingerStat.stateUp += 1;
   }
   
   // Lawmen types
   else if(%char.isOfType("Nicesheriff")) {
      NiceSheriffStat.stateUp += 1;
   }
   else if(%char.isOfType("Toughsheriff")) {
      ToughSheriffStat.stateUp += 1;
   }
   else if(%char.isOfType("Deputy")) {
      DeputyStat.stateUp += 1;
   }
}
//------------------------------------------------------------------------------

//-Object Type Stat Methods-----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
//! Sets the object type stat in the stat gui to a value
//! \param %type Object type set
//! \param %val Value to set
////////////////////////////////////////////////////////////////////////////////
function setObjTypeStat(%type, %val)
{
   %text = %type @ "Stat";
   %text.stateUp = %val;
   
   // update the background button
   %bg = %type @ "BGStat";
   %bg.disabled = (%text.stateUp == 0);
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies the object type stat in the stat gui with a value
//! \param %type Object type modified
//! \param %val Modify value
////////////////////////////////////////////////////////////////////////////////
function modifyObjTypeStat(%type, %val)
{
   %text = %type @ "Stat";
   %text.stateUp += %val;
   
   // update the background button
   %bg = %type @ "BGStat";
   %bg.disabled = (%text.stateUp == 0);
}

////////////////////////////////////////////////////////////////////////////////
//! Saves the object type stat so that it can be zeroed once the gui is closed
//! \param %type Object type
////////////////////////////////////////////////////////////////////////////////
function saveObjTypeStat(%type)
{   
   // If already saved then don't save it
   for(%i = 0; %i < $ObjTypeStatCount; %i++) {
      if($ObjTypeStat[%i] $= %type) {
         return;
      }
   }
   
   // Save stat
   $ObjTypeStat[$ObjTypeStatCount] = %type;
   $ObjTypeStatCount++;
}

////////////////////////////////////////////////////////////////////////////////
//! Resets all saved object type stats to zero and clears the save
////////////////////////////////////////////////////////////////////////////////
function resetObjTypeStats()
{
   // For each object type stored do
   for (%i = 0; %i < $ObjTypeStatCount; %i++)
   {
      %type = $ObjTypeStat[%i];
      
      // reset the state to zero
      setObjTypeStat(%type, 0);
   }
   
   // Reset counter
   $ObjTypeStatCount = 0;
}
//------------------------------------------------------------------------------

//-Happy Modifier Methods-------------------------------------------------------

// remove any mod controls that were dynamically added
function resetHappyMods()
{
   // delete all of the created mod controls
   for (%index = 0; %index < $HappyModCount; %index++)
   {
      // get all three controls that were added for the mod
      %text = "HappyStringStat" @ %index;
      %number = "HappyValueStat" @ %index;
      %icon = "HappyIconStat" @ %index;
      
      // delete the three controls
      %text.delete();
      %number.delete();
      %icon.delete();
   }
   
   // clear the mod count
   $HappyModCount = 0;
}

// if data is gathered for a particularly happy mod, this is the
// happy mod and the data found (data should be added to the stat page)
function clientCmdSetHappyModData(%index, %data)
{
   // update temp positions
   %posX = getWord(HappyStringStatFirst.position, 0);
   %posY = getWord(HappyStringStatFirst.position, 1);
   %diffY = getWord(HappyStringStatNext.position, 1) - %posY;
   
   // update the string that will be used (and check if it is
   // dynamic, if so, update it)
   %dataCount = getWordCount(%data);
   %string = $HappyModString[%index];
   %value = %data;
   if (%dataCount > 1)
   {
      %string = slgFormatObjectString(%string,
         getWords(%data, 0, %dataCount - 2));
      %value = getWord(%data, %dataCount - 1);
   }
   
   // create the text that will display
   %text = new SLText("HappyStringStat" @ $HappyModCount)
   {
      stateUp = %string;
      color = HappyStringStatFirst.color;
      input = false;
      visible = true;
      Profile = HappyStringStatFirst.profile;
      HorizSizing = "relative";
      VertSizing = "relative";
      Position = %posX @ " " @ %posY + %diffY * $HappyModCount;
      Extent = HappyStringStatFirst.extent;
   };
   
   // update temp positions
   %posX = getWord(HappyValueStatFirst.position, 0);
   %posY = getWord(HappyValueStatFirst.position, 1);
   %diffY = getWord(HappyValueStatNext.position, 1) - %posY;
   
   // create the value that will display
   %number = new SLText("HappyValueStat" @ $HappyModCount)
   {
      stateUp = %value;
      color = HappyValueStatFirst.color;
      input = false;
      visible = true;
      Profile = HappyValueStatFirst.profile;
      HorizSizing = "relative";
      VertSizing = "relative";
      Position = %posX @ " " @ %posY + %diffY * $HappyModCount;
      Extent = HappyValueStatFirst.extent;
   };
   
   // update temp positions
   %posX = getWord(HappyIconStatFirst.position, 0);
   %posY = getWord(HappyIconStatFirst.position, 1);
   %diffY = getWord(HappyIconStatNext.position, 1) - %posY;
   
   // create the value that will display
   %icon = new SLImage("HappyIconStat" @ $HappyModCount)
   {
      color = HappyIconStatFirst.color;
      input = false;
      visible = true;
      Profile = HappyIconStatFirst.profile;
      HorizSizing = "relative";
      VertSizing = "relative";
      Position = %posX @ " " @ %posY + %diffY * $HappyModCount;
      Extent = HappyIconStatFirst.extent;
   };
   if (%value > 0) %icon.stateUp = $HappyModIconHappy;
   else %icon.stateUp = $HappyModIconMad;
   
   // update the happiness bonus state
   HappyBonusStat.stateUp += %value;
   
   // add the objects that were created to the stat page
   StatScrollControl.addGuiControl(%text);
   StatScrollControl.addGuiControl(%number);
   StatScrollControl.addGuiControl(%icon);
   
   // increase the happy mod count
   $HappyModCount++;
   
   // find the hud and scroll heights
   %scrollHeight = getWord(BaseStatScrollControl.extent, 1) - 5;
   %hudHeight = getWord(%icon.position, 1) + getWord(%icon.extent, 1);
   
   // no scroll bar
   if (%hudHeight <= %scrollHeight)
   {
      // turn off scroll bars
      BaseStatScrollControl.vScrollBar = "alwaysOff";
      
      // update the hud extent
      StatScrollControl.extent = BaseStatScrollControl.extent;
   }
   // add a scroll bar
   else
   {
      // turn on scroll bars
      BaseStatScrollControl.vScrollBar = "alwaysOn";
      
      // update the hud extent
      StatScrollControl.extent = getWord(BaseStatScrollControl.extent, 0) @ " " @ %hudHeight;
   }
}

// when the server happy mod information has all been received, this
// message is received, and the canvas can be shown
function clientCmdSetHappyModDone()
{
   // update the base happiness from the current and bonus happiness
   HappyStartStat.stateUp = HappyPercentStat.stateUp -
      HappyBonusStat.stateUp;
}


/*
////////////////////////////////////////////////////////////////////////////////
//! Indicates if a happy modifier is currently showing in the gui
//! \param %mod Happy modifier
//! \retval bool True if modifier is currently showing in the gui
////////////////////////////////////////////////////////////////////////////////
function isHappyModInEffect(%mod)
{
   return ($HappyModIndex[%mod] != -1);
}

////////////////////////////////////////////////////////////////////////////////
//! Removes a happy modifier from the gui and resets its value to zero
//! \param %mod Happy modifier
////////////////////////////////////////////////////////////////////////////////
function deleteHappyMod(%mod)
{
   if(!isHappyModInEffect(%mod)) {
      return;
   }
   
   // Move strings and values up in array, erasing happy modifier
   for(%i = $HappyModIndex[%mod]; %i < $HappyModIndexCount - 1; %i++) {
      getHappyModStringGui(%i).stateUp = getHappyModStringGui(%i + 1).stateUp;
      getHappyModValueGui(%i).stateUp  = getHappyModValueGui(%i + 1).stateUp;
   }
   getHappyModStringGui($HappyModIndexCount - 1).stateUp = "";
   getHappyModValueGui($HappyModIndexCount - 1).stateUp = "";
   $HappyModIndexCount--;
   
   // Reset all necessary happy mod indexes
   for(%i = 0; %i < $HMOD_COUNT; %i++) {
      if($HappyModIndex[%i] > $HappyModIndex[%mod]) {
         $HappyModIndex[%i]--;
      }
   }
   $HappyModIndex[%mod] = -1;
}

////////////////////////////////////////////////////////////////////////////////
//! Checks if the happy modifier is non-zero and, if so, updates the stat gui
//! \param %mod Happy modifier
////////////////////////////////////////////////////////////////////////////////
function updateHappyModGui(%mod)
{
    // Check building mod
   if(getHappyModValue(%mod) != 0) {
      setHappyModGui(%mod);
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Adds a happy modifier to the stat screen with the current settings
//! \param %mod Happy modifier
////////////////////////////////////////////////////////////////////////////////
function setHappyModGui(%mod)
{
   // Set value only
   if(isHappyModInEffect(%mod)) {
      setHappyModGuiValue(%mod);
      return;
   }
   
   // Set gui elements for new happy modifier
   %index = $HappyModIndexCount;
   getHappyModStringGui(%index).stateUp = $HappyModString[%mod];
   getHappyModValueGui(%index).stateUp  = getHappyModValue(%mod);
   
   // Save new index for happy modifier
   $HappyModIndex[%mod] = $HappyModIndexCount;
   $HappyModIndexCount++;
}

////////////////////////////////////////////////////////////////////////////////
//! Sets the happy modifier's value in the stat screen with the current setting
//! \param %mod Happy Modifier
////////////////////////////////////////////////////////////////////////////////
function setHappyModGuiValue(%mod)
{
   if(!isHappyModInEffect(%mod)) {
      return;
   }
   
   // Set value
   getHappyModValueGui($HappyModIndex[%mod]).stateUp = getHappyModValue(%mod);
}

////////////////////////////////////////////////////////////////////////////////
//! Resets the happy modifier gui elements and the happy modifier values
////////////////////////////////////////////////////////////////////////////////
function resetHappyMods()
{
   for(%i = 0; %i < $HMOD_COUNT; %i++) {
      // Reset values
      setHappyModValue(%i, 0);
      
      // Reset gui elements
      getHappyModValueGui(%i).stateUp  = "";
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the happy modifier value
//! \param %mod Happy modifier
////////////////////////////////////////////////////////////////////////////////
function getHappyModValue(%mod)
{
   return $HappyModValue[%mod];
}

////////////////////////////////////////////////////////////////////////////////
//! Sets the happy modifier value
//! \param %mod Happy modifier
//! \param %val New value for modifier
////////////////////////////////////////////////////////////////////////////////
function setHappyModValue(%mod, %val)
{
   $HappyModValue[%mod] = %val;
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies the happy modifier value
//! \param %mod Happy modifier
//! \param %val Modify value
////////////////////////////////////////////////////////////////////////////////
function modifyHappyModValue(%mod, %val)
{
   $HappyModValue[%mod] += %val;
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the name of the happy modifier string gui element
//! \param %index Index of gui element
////////////////////////////////////////////////////////////////////////////////
function getHappyModStringGui(%index)
{
   return ("HappyStringStat" @ %index);
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the name of the happy modifier value gui element
//! \param %index Index of gui element
////////////////////////////////////////////////////////////////////////////////
function getHappyModValueGui(%index)
{
   return ("HappyValueStat" @ %index);
}
*/

////////////////////////////////////////////////////////////////////////////////
//! Sets the happy modifier values received from the server
//! \param %mods Modifiers to set
//! \param %vals Values for modifiers
////////////////////////////////////////////////////////////////////////////////
function clientCmdSetHappyMods(%mods, %vals)
{
   /*
   // Set the value for each happy modifier
   %count = getWordCount(%mods);
   for(%i = 0; %i < %count; %i++) {
      setHappyModValue(getWord(%mods, %i), getWord(%vals, %i));
   }
   */
   
   // show gui
   openStatGui();
}

//------------------------------------------------------------------------------
// End client_statGui.cs